
const DEFAULTS = { durationSec: 15, streams: 4, uploadSizeMB: 8, latencySamples: 10 };

const els = {
  duration: document.getElementById('duration'),
  streams: document.getElementById('streams'),
  uploadSize: document.getElementById('uploadSize'),
  latencySamples: document.getElementById('latencySamples'),
  startBtn: document.getElementById('startBtn'),
  latency: document.getElementById('latency'),
  jitter: document.getElementById('jitter'),
  dlMbps: document.getElementById('dlMbps'),
  ulMbps: document.getElementById('ulMbps'),
  gauge: document.getElementById('gauge'),
  gaugeVal: document.getElementById('gaugeVal'),
  serverLabel: document.getElementById('serverLabel'),
};
const sleep=(ms)=>new Promise(r=>setTimeout(r,ms));
const now=()=>performance.now();
const clamp=(x,min,max)=>Math.max(min,Math.min(max,x));
const bytesToMbps=(bytes,sec)=>sec>0?(bytes*8)/(sec*1e6):0;
const mean=a=>a.reduce((x,y)=>x+y,0)/(a.length||1);
const std=a=>{const m=mean(a);return Math.sqrt(mean(a.map(x=>(x-m)**2)));};

function getServer(){
  try{ const s=JSON.parse(localStorage.getItem('vc_server')); if(s&&s.base){return s;} }catch(e){}
  return { name:'Default', base:'', label:'Default' };
}
function setServerLabel(){ const s=getServer(); els.serverLabel && (els.serverLabel.textContent = s.name || 'Default'); }
setServerLabel();

// Gauge draw
const ctx = els.gauge.getContext('2d');
function drawGauge(valueMbps){
  const c=ctx, W=els.gauge.width, H=els.gauge.height, R=W/2-18;
  c.clearRect(0,0,W,H);
  c.save(); c.translate(W/2,H/2);
  const max=1000; // scale up to 1 Gbps for the dial
  const frac=Math.min(1, valueMbps/max);
  // track
  c.beginPath(); c.strokeStyle='rgba(255,255,255,.12)'; c.lineWidth=18; c.arc(0,0,R,Math.PI*0.75,Math.PI*0.25,false); c.stroke();
  // progress
  c.beginPath(); const grad=c.createLinearGradient(-R,0,R,0); grad.addColorStop(0,'#6ee7ff'); grad.addColorStop(1,'#7c3aed');
  c.strokeStyle=grad; c.lineWidth=18; c.lineCap='round';
  c.arc(0,0,R,Math.PI*0.75, Math.PI*0.75 + frac*(Math.PI*1.5), false); c.stroke();
  // ticks
  c.rotate(Math.PI*0.75);
  c.strokeStyle='rgba(255,255,255,.18)'; c.lineWidth=2;
  for(let i=0;i<=10;i++){ c.beginPath(); c.moveTo(R-8,0); c.lineTo(R,0); c.stroke(); c.rotate(Math.PI*1.5/10); }
  c.restore();
  els.gaugeVal.textContent = Math.round(valueMbps);
}

// Core tests (ping/download/upload) – same server, optional base override
function mkURL(path){ const base=getServer().base || ''; return base ? (base.replace(/\/+$/,'') + '/' + path) : path; }

async function measureLatency(n){
  const t=[];
  for(let i=0;i<n;i++){
    const t0=now();
    try{ const r=await fetch(mkURL('api/ping.php?i='+i+'&ts='+t0),{cache:'no-store'}); await r.text(); }catch(e){}
    t.push(now()-t0);
    drawGauge((i+1)/n*50);
    await sleep(60);
  }
  const s=[...t].sort((a,b)=>a-b); const trimmed=s.slice(0,Math.max(1,s.length-1));
  return {avg:mean(trimmed), jitter:std(trimmed)};
}

async function downloadTest({streams,durationSec},onP){
  const aborts=[]; let total=0; const start=now();
  function run(i){
    const ctl=new AbortController(); aborts.push(ctl);
    return (async()=>{
      try{
        const r=await fetch(mkURL('api/download.php?stream='+i+'&ts='+Date.now()),{cache:'no-store',signal:ctl.signal});
        if(!r.body)return; const rd=r.body.getReader();
        while(true){ const {value,done}=await rd.read(); if(done)break; total+=value.length; onP&&onP(total,(now()-start)/1000); }
      }catch(e){}
    })();
  }
  const tasks=[]; for(let i=0;i<streams;i++) tasks.push(run(i));
  await sleep(durationSec*1000); aborts.forEach(a=>a.abort()); await Promise.allSettled(tasks);
  return {bytes:total, seconds:(now()-start)/1000};
}

function randomBlob(n){ const a=new Uint8Array(n); (crypto&&crypto.getRandomValues)?crypto.getRandomValues(a):a.forEach((_,i)=>a[i]=(Math.random()*256)|0); return new Blob([a]); }
async function uploadTest({streams,durationSec,uploadSizeMB},onP){
  const payload=randomBlob(uploadSizeMB*1024*1024); let total=0; const start=now(); const end=start+durationSec*1000;
  function send(){ return new Promise(res=>{ const x=new XMLHttpRequest(); x.open('POST', mkURL('api/upload.php?ts='+Date.now())); x.onload=()=>{ total+=payload.size; onP&&onP(total,(now()-start)/1000); res(true); }; x.onerror=()=>res(false); x.send(payload); }); }
  async function loop(){ while(now()<end){ const ok=await send(); if(!ok) await sleep(120);} }
  const tasks=[]; for(let i=0;i<streams;i++) tasks.push(loop()); await Promise.all(tasks);
  return {bytes:total, seconds:(now()-start)/1000};
}

function saveResult(ping,dl,ul,server){
  const rec={ t:new Date().toISOString(), ping, dl, ul, server };
  const key='vc_results'; const all=JSON.parse(localStorage.getItem(key)||'[]'); all.unshift(rec); localStorage.setItem(key,JSON.stringify(all.slice(0,200)));
}

function setBusy(b){ els.startBtn.disabled=b; ['duration','streams','uploadSize','latencySamples'].forEach(id=>{const e=document.getElementById(id); if(e) e.disabled=b;}); }

async function run(){
  setBusy(true);
  drawGauge(0);
  const cfg={ durationSec:clamp(parseInt(els.duration.value||DEFAULTS.durationSec),5,60),
              streams:clamp(parseInt(els.streams.value||DEFAULTS.streams),1,10),
              uploadSizeMB:clamp(parseInt(els.uploadSize.value||DEFAULTS.uploadSizeMB),1,64),
              latencySamples:clamp(parseInt(els.latencySamples.value||DEFAULTS.latencySamples),5,30) };
  try{
    const {avg,jitter}=await measureLatency(cfg.latencySamples);
    els.latency.textContent=avg.toFixed(0); els.jitter.textContent=jitter.toFixed(0);
    let cur=0, peak=0;
    const dl=await downloadTest({streams:cfg.streams,durationSec:cfg.durationSec},(bytes,secs)=>{ cur=bytesToMbps(bytes,Math.max(0.5,secs)); peak=Math.max(peak,cur); drawGauge(peak); els.dlMbps.textContent=cur.toFixed(1); });
    const finalDL=bytesToMbps(dl.bytes,dl.seconds); els.dlMbps.textContent=finalDL.toFixed(1); drawGauge(finalDL);
    const ul=await uploadTest({streams:cfg.streams,durationSec:cfg.durationSec,uploadSizeMB:cfg.uploadSizeMB},(bytes,secs)=>{ cur=bytesToMbps(bytes,Math.max(0.5,secs)); peak=Math.max(peak,cur); drawGauge(peak); els.ulMbps.textContent=cur.toFixed(1); });
    const finalUL=bytesToMbps(ul.bytes,ul.seconds); els.ulMbps.textContent=finalUL.toFixed(1); drawGauge(finalUL);
    saveResult(parseFloat(els.latency.textContent), finalDL, finalUL, (getServer().name||'Default'));
  }catch(e){ alert('Error: '+(e?.message||e)); } finally{ setBusy(false); }
}
document.getElementById('startBtn').addEventListener('click',run);
drawGauge(0);
